<?php
namespace App\Controllers;

use App\Core\Controller;
use App\Core\Helpers;
use App\Core\Marketplace;

class AdminController extends Controller
{
    private Marketplace $market;

    public function __construct(array $config, $storage, $auth, $security)
    {
        parent::__construct($config, $storage, $auth, $security);
        $this->market = new Marketplace($storage);
    }

    public function dashboard(): void
    {
        $this->auth->requireRole(['admin', 'superadmin']);
        $this->view('admin/dashboard', ['title' => 'Admin', 'stats' => $this->market->stats()]);
    }

    public function listings(): void
    {
        $this->auth->requireRole(['admin', 'superadmin']);
        $status = $_GET['estado'] ?? 'pending';
        $items = $this->market->searchListings(['status' => $status === 'todos' ? 'all' : $status]);
        $this->view('admin/listings', ['title' => 'Moderación', 'items' => $items, 'status' => $status]);
    }

    public function approve(): void
    {
        $this->auth->requireRole(['admin', 'superadmin']);
        if (!$this->security->verifyCsrf($_POST['_csrf'] ?? null)) Helpers::redirect('/admin/publicaciones');
        $id = $_GET['id'] ?? '';
        $item = $this->market->getListing($id);
        if ($item) {
            $item['status'] = 'approved';
            $item['moderation']['approvedBy'] = $this->auth->user()['id'];
            $item['moderation']['rejectedReason'] = null;
            // Asignar modo de contacto por defecto si no tiene uno
            if (empty($item['contactMode'])) {
                $settings = $this->storage->readJson('pages/settings.json', []);
                $item['contactMode'] = $settings['default_contact_mode'] ?? 'seller';
            }
            $this->market->saveListing($item);
            $this->notify($item['ownerId'], 'Tu publicación fue aprobada: ' . $item['title']);
            $this->storage->appendLog('admin.log', 'APPROVED ' . $id . ' by ' . $this->auth->user()['id']);
        }
        Helpers::redirect('/admin/publicaciones?estado=pending');
    }

    public function reject(): void
    {
        $this->auth->requireRole(['admin', 'superadmin']);
        if (!$this->security->verifyCsrf($_POST['_csrf'] ?? null)) Helpers::redirect('/admin/publicaciones');
        $id = $_GET['id'] ?? '';
        $item = $this->market->getListing($id);
        if ($item) {
            $item['status'] = 'rejected';
            $item['moderation']['approvedBy'] = null;
            $item['moderation']['rejectedReason'] = trim($_POST['reason'] ?? 'Sin detalle');
            $this->market->saveListing($item);
            $this->notify($item['ownerId'], 'Tu publicación fue rechazada: ' . $item['moderation']['rejectedReason']);
            $this->storage->appendLog('admin.log', 'REJECTED ' . $id . ' by ' . $this->auth->user()['id']);
        }
        Helpers::redirect('/admin/publicaciones?estado=pending');
    }

    public function markSold(): void
    {
        $this->auth->requireRole(['superadmin']);
        if (!$this->security->verifyCsrf($_POST['_csrf'] ?? null)) Helpers::redirect('/admin/publicaciones');
        $id = $_GET['id'] ?? '';
        $item = $this->market->getListing($id);
        if ($item && ($item['status'] ?? '') === 'approved') {
            $item['status'] = 'sold';
            $item['soldAt'] = time();
            $item['soldBy'] = $this->auth->user()['id'];
            $this->market->saveListing($item);
            $this->notify($item['ownerId'], 'Tu producto fue marcado como vendido: ' . $item['title']);
            $this->storage->appendLog('admin.log', 'MARKED_SOLD ' . $id . ' by ' . $this->auth->user()['id']);
            Helpers::flash('success', 'Publicación marcada como VENDIDO.');
        }
        Helpers::redirect('/admin/publicaciones?estado=approved');
    }

    public function editListing(): void
    {
        $this->auth->requireRole(['admin', 'superadmin']);
        if (!$this->security->verifyCsrf($_POST['_csrf'] ?? null)) Helpers::redirect('/admin/publicaciones');
        $id = $_GET['id'] ?? '';
        $item = $this->market->getListing($id);
        if ($item) {
            $item['title'] = trim($_POST['title'] ?? $item['title']);
            $item['category'] = trim($_POST['category'] ?? $item['category']);
            $item['featured'] = isset($_POST['featured']);
            $item['priority'] = max(0, (int)($_POST['priority'] ?? 0));
            // Status: solo superadmin puede poner "sold"
            $newStatus = trim($_POST['status'] ?? $item['status']);
            if ($newStatus === 'sold' && ($this->auth->user()['role'] ?? '') !== 'superadmin') {
                $newStatus = $item['status'];
            }
            $item['status'] = $newStatus;
            // Modo de contacto: seller o mediator
            $item['contactMode'] = in_array($_POST['contact_mode'] ?? '', ['seller', 'mediator'])
                ? $_POST['contact_mode'] : ($item['contactMode'] ?? 'seller');
            $this->market->saveListing($item);
            $this->storage->appendLog('admin.log', 'EDITED ' . $id . ' by ' . $this->auth->user()['id']);
        }
        Helpers::redirect('/admin/publicaciones?estado=todos');
    }

    public function users(): void
    {
        $this->auth->requireRole(['admin', 'superadmin']);
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            if (!$this->security->verifyCsrf($_POST['_csrf'] ?? null)) Helpers::redirect('/admin/usuarios');
            $id = $_POST['id'] ?? '';
            $profile = $this->storage->readJson('users/' . $id . '/profile.json', []);
            if ($profile) {
                $profile['status'] = ($_POST['action'] ?? '') === 'block' ? 'blocked' : 'active';
                $this->storage->writeJson('users/' . $id . '/profile.json', $profile);
                $idx = $this->storage->readJson('users/users.json', ['items' => []]);
                foreach ($idx['items'] as &$u) if ($u['id'] === $id) $u['status'] = $profile['status'];
                $this->storage->writeJson('users/users.json', $idx);
            }
            Helpers::redirect('/admin/usuarios');
        }
        $users = $this->storage->readJson('users/users.json', ['items' => []])['items'] ?? [];
        $this->view('admin/users', ['title' => 'Usuarios', 'users' => $users]);
    }

    public function categories(): void
    {
        $this->auth->requireRole(['admin', 'superadmin']);
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            if (!$this->security->verifyCsrf($_POST['_csrf'] ?? null)) Helpers::redirect('/admin/categorias');
            $cats = $this->market->getCategories();
            $cats[] = [
                'id' => Helpers::randomId('c'),
                'name' => trim($_POST['name'] ?? ''),
                'slug' => Helpers::slug($_POST['name'] ?? ''),
                'visible' => true,
            ];
            $this->market->saveCategories($cats);
            Helpers::redirect('/admin/categorias');
        }
        $this->view('admin/categories', ['title' => 'Categorías', 'categories' => $this->market->getCategories()]);
    }

    public function ads(): void
    {
        $this->auth->requireRole(['admin', 'superadmin']);
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            if (!$this->security->verifyCsrf($_POST['_csrf'] ?? null)) Helpers::redirect('/admin/publicidad');
            $slots = $this->market->ads();
            $slots[] = [
                'key' => trim($_POST['key'] ?? ''),
                'type' => $_POST['type'] ?? 'banner',
                'enabled' => isset($_POST['enabled']),
                'code' => trim($_POST['code'] ?? ''),
            ];
            $this->market->saveAds($slots);
            Helpers::redirect('/admin/publicidad');
        }
        $this->view('admin/ads', ['title' => 'Publicidad', 'slots' => $this->market->ads()]);
    }

    public function pages(): void
    {
        $this->auth->requireRole(['admin', 'superadmin']);
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            if (!$this->security->verifyCsrf($_POST['_csrf'] ?? null)) Helpers::redirect('/admin/paginas');
            $pages = $this->market->pages();
            $pages[] = [
                'id' => Helpers::randomId('p'),
                'title' => trim($_POST['title'] ?? ''),
                'slug' => Helpers::slug($_POST['title'] ?? ''),
                'content' => trim($_POST['content'] ?? ''),
                'updatedAt' => time(),
            ];
            $this->market->savePages($pages);
            Helpers::redirect('/admin/paginas');
        }
        $this->view('admin/pages', ['title' => 'Páginas', 'pages' => $this->market->pages()]);
    }

    public function reports(): void
    {
        $this->auth->requireRole(['admin', 'superadmin']);
        $this->view('admin/reports', ['title' => 'Reportes', 'reports' => $this->market->reports()]);
    }

    public function logs(): void
    {
        $this->auth->requireRole(['admin', 'superadmin']);
        $files = ['app.log', 'auth.log', 'admin.log'];
        $logs = [];
        foreach ($files as $f) {
            $path = $this->storage->path('logs/' . $f);
            $logs[$f] = file_exists($path) ? array_slice(file($path, FILE_IGNORE_NEW_LINES), -200) : [];
        }
        $this->view('admin/logs', ['title' => 'Logs', 'logs' => $logs]);
    }

    public function rebuildIndexes(): void
    {
        $this->auth->requireRole(['admin', 'superadmin']);
        if (!$this->security->verifyCsrf($_POST['_csrf'] ?? null)) Helpers::redirect('/admin');
        $this->market->rebuildListingIndex();
        Helpers::flash('success', 'Índices reconstruidos.');
        Helpers::redirect('/admin');
    }

    private function notify(string $userId, string $message): void
    {
        $file = 'users/' . $userId . '/notifications.json';
        $data = $this->storage->readJson($file, ['items' => []]);
        $data['items'][] = ['id' => Helpers::randomId('n'), 'message' => $message, 'createdAt' => time(), 'read' => false];
        $this->storage->writeJson($file, $data);
    }
}
