<?php
namespace App\Controllers;

use App\Core\Controller;
use App\Core\Helpers;
use App\Core\Marketplace;
use App\Core\Validator;

class AuthController extends Controller
{
    public function registerForm(): void
    {
        $this->view('auth/register', ['title' => 'Registro']);
    }

    public function register(): void
    {
        if (!$this->security->rateLimit('register')) {
            Helpers::flash('error', 'Demasiados intentos. Intenta más tarde.');
            Helpers::redirect('/registro');
        }
        if (!$this->security->verifyCsrf($_POST['_csrf'] ?? null)) {
            Helpers::flash('error', 'CSRF inválido.');
            Helpers::redirect('/registro');
        }

        $v = (new Validator())
            ->required('name', $_POST['name'] ?? '', 'Nombre requerido')
            ->required('email', $_POST['email'] ?? '', 'Email requerido')
            ->email('email', $_POST['email'] ?? '', 'Email inválido')
            ->required('password', $_POST['password'] ?? '', 'Contraseña requerida')
            ->min('password', $_POST['password'] ?? '', 8, 'Mínimo 8 caracteres');

        if ($v->fails()) {
            Helpers::flash('error', implode(' | ', $v->errors()));
            Helpers::redirect('/registro');
        }

        $email = strtolower(trim($_POST['email']));
        $index = $this->storage->readJson('users/users.json', ['items' => []])['items'] ?? [];
        foreach ($index as $item) {
            if (strcasecmp($item['email'], $email) === 0) {
                Helpers::flash('error', 'El email ya existe.');
                Helpers::redirect('/registro');
            }
        }

        $id = $this->auth->register([
            'name' => trim($_POST['name']),
            'email' => $email,
            'phone' => trim($_POST['phone'] ?? ''),
            'whatsapp' => trim($_POST['whatsapp'] ?? ''),
            'location' => trim($_POST['location'] ?? ''),
            'password' => $_POST['password'],
        ]);
        $_SESSION['user_id'] = $id;
        Helpers::flash('success', 'Cuenta creada. Bienvenido.');
        Helpers::redirect('/panel');
    }

    public function loginForm(): void
    {
        $this->view('auth/login', ['title' => 'Iniciar sesión']);
    }

    public function login(): void
    {
        if (!$this->security->rateLimit('login')) {
            Helpers::flash('error', 'Demasiados intentos de login.');
            Helpers::redirect('/login');
        }
        if (!$this->security->verifyCsrf($_POST['_csrf'] ?? null)) {
            Helpers::flash('error', 'CSRF inválido.');
            Helpers::redirect('/login');
        }
        if (!$this->auth->attempt(trim($_POST['email'] ?? ''), $_POST['password'] ?? '')) {
            Helpers::flash('error', 'Credenciales inválidas');
            Helpers::redirect('/login');
        }
        Helpers::redirect('/panel');
    }

    public function logout(): void
    {
        if (!$this->security->verifyCsrf($_POST['_csrf'] ?? null)) {
            Helpers::redirect('/');
        }
        $this->auth->logout();
        Helpers::redirect('/');
    }

    public function recoverForm(): void
    {
        $this->view('auth/recover', ['title' => 'Recuperar contraseña']);
    }

    public function recover(): void
    {
        if (!$this->security->verifyCsrf($_POST['_csrf'] ?? null)) {
            Helpers::flash('error', 'CSRF inválido.');
            Helpers::redirect('/recuperar');
        }
        $email = strtolower(trim($_POST['email'] ?? ''));
        $users = $this->storage->readJson('users/users.json', ['items' => []])['items'] ?? [];
        foreach ($users as $u) {
            if (strcasecmp($u['email'], $email) === 0) {
                $sec = $this->storage->readJson('users/' . $u['id'] . '/security.json', []);
                $token = bin2hex(random_bytes(24));
                $sec['resetTokens'][] = ['token' => $token, 'expiresAt' => time() + 3600, 'used' => false];
                $this->storage->writeJson('users/' . $u['id'] . '/security.json', $sec);
                Helpers::flash('success', 'Token generado (demo): ' . $token);
                Helpers::redirect('/reset?token=' . $token);
            }
        }
        Helpers::flash('error', 'No se encontró el correo.');
        Helpers::redirect('/recuperar');
    }

    public function resetForm(): void
    {
        $this->view('auth/reset', ['title' => 'Reset contraseña', 'token' => $_GET['token'] ?? '']);
    }

    public function reset(): void
    {
        if (!$this->security->verifyCsrf($_POST['_csrf'] ?? null)) {
            Helpers::flash('error', 'CSRF inválido.');
            Helpers::redirect('/recuperar');
        }
        $token = trim($_POST['token'] ?? '');
        $pass = $_POST['password'] ?? '';
        if (strlen($pass) < 8) {
            Helpers::flash('error', 'Contraseña muy corta.');
            Helpers::redirect('/reset?token=' . urlencode($token));
        }

        $users = $this->storage->readJson('users/users.json', ['items' => []])['items'] ?? [];
        foreach ($users as $u) {
            $sec = $this->storage->readJson('users/' . $u['id'] . '/security.json', []);
            foreach ($sec['resetTokens'] ?? [] as &$rt) {
                if ($rt['token'] === $token && !$rt['used'] && $rt['expiresAt'] > time()) {
                    $sec['passwordHash'] = password_hash($pass, PASSWORD_BCRYPT);
                    $sec['lastPasswordChange'] = time();
                    $rt['used'] = true;
                    $this->storage->writeJson('users/' . $u['id'] . '/security.json', $sec);
                    Helpers::flash('success', 'Contraseña actualizada.');
                    Helpers::redirect('/login');
                }
            }
        }
        Helpers::flash('error', 'Token inválido o expirado.');
        Helpers::redirect('/recuperar');
    }
}
