<?php
namespace App\Controllers;

use App\Core\Controller;
use App\Core\Helpers;
use App\Core\Marketplace;

class PublicController extends Controller
{
    private Marketplace $market;

    public function __construct(array $config, $storage, $auth, $security)
    {
        parent::__construct($config, $storage, $auth, $security);
        $this->market = new Marketplace($storage);
    }

    public function home(): void
    {
        $cacheKey = 'cache/home.json';
        $cache = $this->storage->readJson($cacheKey, []);
        if (!empty($cache['expiresAt']) && $cache['expiresAt'] > time()) {
            $items = $cache['items'] ?? [];
        } else {
            $items = array_slice($this->market->searchListings(['status' => 'visible']), 0, 16);
            $this->storage->writeJson($cacheKey, ['items' => $items, 'expiresAt' => time() + 120]);
        }
        $vehicles = array_values(array_filter($items, fn($x) => in_array($x['category'], ['autos', 'motos', 'motores', 'repuestos', 'llantas', 'faros', 'pernos', 'valvulas'], true)));
        $this->view('public/home', [
            'title' => 'Inicio',
            'items' => $items,
            'vehicles' => array_slice($vehicles, 0, 8),
            'categories' => $this->market->getCategories(),
            'ads' => $this->market->ads(),
        ]);
    }

    public function search(): void
    {
        $filters = [
            'q' => $_GET['q'] ?? '',
            'cat' => $_GET['cat'] ?? '',
            'min' => $_GET['min'] ?? '',
            'max' => $_GET['max'] ?? '',
            'cond' => $_GET['cond'] ?? '',
            'loc' => $_GET['loc'] ?? '',
            'status' => ($_GET['status'] ?? 'visible'),
            'type' => $_GET['type'] ?? '',
            'shipping' => $_GET['shipping'] ?? '',
            'verified' => $_GET['verified'] ?? '',
        ];
        $all = $this->market->searchListings($filters);
        $page = max(1, (int)($_GET['page'] ?? 1));
        $per = $this->config['app']['items_per_page'];
        $offset = ($page - 1) * $per;
        $items = array_slice($all, $offset, $per);

        $this->view('public/search', [
            'title' => 'Buscar',
            'items' => $items,
            'total' => count($all),
            'page' => $page,
            'per' => $per,
            'filters' => $filters,
            'categories' => $this->market->getCategories(),
        ]);
    }

    public function category(array $params): void
    {
        $_GET['cat'] = $params['slug'] ?? '';
        $this->search();
    }

    public function product(array $params): void
    {
        $item = $this->market->getListing($params['id']);
        if (!$item || !in_array($item['status'] ?? '', ['approved', 'sold'])) {
            http_response_code(404);
            echo 'Producto no disponible';
            return;
        }
        $this->market->incrementViews($params['id']);
        $owner = $this->storage->readJson('users/' . $item['ownerId'] . '/profile.json', []);

        // Resolver contacto: mediador vs vendedor vs vendido
        $contactMode = $item['contactMode'] ?? 'seller';
        $settings = $this->storage->readJson('pages/settings.json', []);
        $isSold = ($item['status'] ?? '') === 'sold';

        if ($isSold) {
            $contactPhone = '';
            $contactWhatsapp = '';
            $contactName = '';
        } elseif ($contactMode === 'mediator' && !empty($settings['mediator_phone'])) {
            $contactPhone = $settings['mediator_phone'];
            $contactWhatsapp = $settings['mediator_whatsapp'] ?? $settings['mediator_phone'];
            $contactName = 'Mediador (Taller Mojarra)';
        } else {
            $contactPhone = $owner['phone'] ?? '';
            $contactWhatsapp = $owner['whatsapp'] ?? $owner['phone'] ?? '';
            $contactName = $owner['name'] ?? 'Vendedor';
        }

        $this->view('public/product', [
            'title' => $item['title'],
            'item' => $item,
            'owner' => $owner,
            'contactPhone' => $contactPhone,
            'contactWhatsapp' => $contactWhatsapp,
            'contactName' => $contactName,
            'contactMode' => $contactMode,
            'isSold' => $isSold,
        ]);
    }

    public function report(array $params): void
    {
        if (!$this->security->verifyCsrf($_POST['_csrf'] ?? null)) {
            Helpers::flash('error', 'CSRF inválido.');
            Helpers::redirect('/producto/' . $params['listingId'] . '/item');
        }
        $this->market->saveReport([
            'listingId' => $params['listingId'],
            'reason' => trim($_POST['reason'] ?? ''),
            'comment' => trim($_POST['comment'] ?? ''),
            'evidence' => '',
            'reporterId' => $this->auth->user()['id'] ?? null,
        ]);
        Helpers::flash('success', 'Reporte enviado al equipo de moderación.');
        Helpers::redirect('/producto/' . $params['listingId'] . '/item');
    }

    public function page(array $params): void
    {
        $pages = $this->market->pages();
        foreach ($pages as $page) {
            if (($page['slug'] ?? '') === ($params['slug'] ?? '')) {
                $this->view('public/page', ['title' => $page['title'], 'page' => $page]);
                return;
            }
        }
        http_response_code(404);
        echo 'Página no encontrada';
    }

    public function favorite(array $params): void
    {
        if (!$this->auth->check()) {
            Helpers::flash('error', 'Debes iniciar sesión para guardar favoritos.');
            Helpers::redirect('/login');
        }
        if (!$this->security->verifyCsrf($_POST['_csrf'] ?? null)) {
            Helpers::redirect('/');
        }
        $listingId = $params['listingId'] ?? '';
        $userId = $this->auth->user()['id'];
        $file = 'users/' . $userId . '/favorites.json';
        $fav = $this->storage->readJson($file, ['items' => []]);
        if (!in_array($listingId, $fav['items'], true)) {
            $fav['items'][] = $listingId;
            $this->storage->writeJson($file, $fav);
        }
        Helpers::flash('success', 'Agregado a favoritos.');
        Helpers::redirect('/producto/' . $listingId . '/item');
    }
}
